"use client";

import React, { createContext, useContext, useEffect, useState } from "react";
import axios from "axios";

const API_BASE_URL = process.env.NEXT_PUBLIC_API_URL || "http://localhost:5001/api";

interface SettingsContextType {
    timeFormat: "12" | "24";
    refreshSettings: () => Promise<void>;
}

const SettingsContext = createContext<SettingsContextType>({
    timeFormat: "12",
    refreshSettings: async () => { },
});

export const useSettings = () => useContext(SettingsContext);

export function SettingsProvider({ children }: { children: React.ReactNode }) {
    const [timeFormat, setTimeFormat] = useState<"12" | "24">("12");

    const fetchSettings = React.useCallback(async () => {
        const token = localStorage.getItem("token");
        if (!token) return;
        try {
            const response = await axios.get(`${API_BASE_URL}/settings`, {
                headers: { 'Authorization': `Bearer ${token}` }
            });
            if (response.data?.status === "success") {
                const settings = response.data.data.settings;
                setTimeFormat(settings.timeFormat || "12");
            }
        } catch (err) {
            console.error("Failed to load settings in provider", err);
        }
    }, []);

    useEffect(() => {
        fetchSettings();
    }, []);

    return (
        <SettingsContext.Provider value={{ timeFormat, refreshSettings: fetchSettings }}>
            {children}
        </SettingsContext.Provider>
    );
}

export function formatTime(dateInput: string | Date | undefined, format: "12" | "24") {
    if (!dateInput) return "";
    const date = new Date(dateInput);
    if (isNaN(date.getTime())) return "";

    return date.toLocaleTimeString([], {
        hour: '2-digit',
        minute: '2-digit',
        hour12: format === "12"
    });
}
